#include <vector>
#include <stdlib.h>
#include <iostream>
#include <cmath>
#include <cstring>
#include <algorithm>
#include <fstream>
#include <memory>
#include <random>
#include <string>
#include <sstream>


using namespace std;

class VertexBahmani {
    public:
    
    int part;
    bool is_deleted; 
    int current_pass_degree;
    int id;

    VertexBahmani(int id, int part) {
        this->id = id;
        this->part = part;
        this->is_deleted = false;
        this->current_pass_degree = 0;
    };

};

class GraphBahmani {
    public:
    vector<shared_ptr<VertexBahmani>> leftVertices, rightVertices;
    vector<shared_ptr<VertexBahmani>> remainedLeftVertices, remainedRightVertices;
    int n1 = 0; int n2 = 0; int remaining_edges = 0;

    GraphBahmani(int n1, int n2) {
        this->n1 = n1;
        this->n2 = n2;
        for(int i = 0; i < n1; i++){
            auto v = std::make_shared<VertexBahmani>(i, 0);
            leftVertices.push_back(v);
        }
        for(int i = 0; i < n2; i++){
            auto v = std::make_shared<VertexBahmani>(i, 1);
            rightVertices.push_back(v);
        }
    };

    void reset_remaining_vertices() {
        remainedLeftVertices.clear();
        remainedRightVertices.clear();
        for (const auto& vertex : leftVertices) {
            vertex->is_deleted = false;
            vertex->current_pass_degree = 0;
            remainedLeftVertices.push_back(vertex);
        }
        remainedRightVertices.clear();
        for (const auto& vertex : rightVertices) {
            vertex->is_deleted = false;
            vertex->current_pass_degree = 0;
            remainedRightVertices.push_back(vertex);
        }
        remaining_edges = 0;
    };

    void reset_remaining_degrees() {
        remaining_edges = 0;
        for (int i = 0; i < remainedLeftVertices.size(); i++)
            remainedLeftVertices[i]->current_pass_degree = 0;
        for (int i = 0; i < remainedRightVertices.size(); i++)
            remainedRightVertices[i]->current_pass_degree = 0;
    }

    void remove_A_set(int part, long double epsilon) {
        if (part == 0) {
            vector<shared_ptr<VertexBahmani>> newRemainedLeftVertices;
            for (const auto& vertex : remainedLeftVertices) {
                auto sharedVertex = vertex;
                if (sharedVertex->current_pass_degree <= (1.L + epsilon) * remaining_edges / remainedLeftVertices.size())
                    sharedVertex->is_deleted = true;
                else
                    newRemainedLeftVertices.push_back(sharedVertex);
            }
            remainedLeftVertices = newRemainedLeftVertices;
        } else {
            vector<shared_ptr<VertexBahmani>> newRemainedRightVertices;
            for (const auto& vertex : remainedRightVertices) {
                auto sharedVertex = vertex;
                if (sharedVertex->current_pass_degree <= (1.L + epsilon) * remaining_edges / remainedRightVertices.size())
                    sharedVertex->is_deleted = true;
                else
                    newRemainedRightVertices.push_back(sharedVertex);
            }
            remainedRightVertices = newRemainedRightVertices;
        }
    }
};

class AlgorithmBahmani {
    public:
    
    long double epsilon;
    long double delta;
    string test_case;
    string algorithm_name = "MPCBahmaniAlgorithm";
    GraphBahmani* G;
    vector<pair<int, int>*> stream_edges;

    AlgorithmBahmani(GraphBahmani* G,  long double delta, long double epsilon, string test_case, vector<std::pair<int, int>*> stream_edges){
        this->G = G;
        this->epsilon = epsilon;
        this->delta = delta;
        this->test_case = test_case;
        this->stream_edges = stream_edges; // we are not actually saving the edges, but we are using this list to iterate over them in each pass
        std::cout << "AlgorithmBahmani initialized with epsilon: " << epsilon << ", delta: " << delta << ", test_case: " << test_case << "\n";
    }

    void debug(long double z, long double density) {
        cout << "z: " << z << "\n";
        cout << "left remaining vertices: ";
        for (auto u : G->remainedLeftVertices)
            cout << u->id << " ";
        cout << "\n";
        cout << "right remaining vertices: ";
        for (auto u : G->remainedRightVertices)
            cout << u->id << " ";
        cout << "\n";
        cout << "density: " << density;
        cout << "\n\n";
    }

    pair<int, long double> run_for_fixed_z(long double z) {
        int number_of_passes = 0;
        G->reset_remaining_vertices();
        long double best_density = 0.0;
        while (G->remainedLeftVertices.size() > 0 && G->remainedRightVertices.size() > 0) {
            G->reset_remaining_degrees();
            //a pass over the edge stream
            number_of_passes += ceil(1.L / delta); 
            for (int i = 0; i < stream_edges.size(); i++) {
                int u_counter = stream_edges[i]->first, v_counter = stream_edges[i]->second;
                auto u = G->leftVertices[u_counter];
                auto v = G->rightVertices[v_counter];
                if (u->is_deleted == true || v->is_deleted == true)
                    continue;
                u->current_pass_degree++;
                v->current_pass_degree++;
                G->remaining_edges++;
            }
            //density for the subgraphs before removing
            long double current_density = 0.0;
            if(G->remainedLeftVertices.size() > 0 && G->remainedRightVertices.size()> 0)
                current_density = 1.L*G->remaining_edges / (sqrtl(1.L * G->remainedLeftVertices.size() * G->remainedRightVertices.size()));
            best_density = max(best_density, current_density);

            //removing the vertices
            if (G->remainedRightVertices.size() == 0 || 1.L * G->remainedLeftVertices.size() >= z * G->remainedRightVertices.size()) {
                G->remove_A_set(0, epsilon);
            } else {
                G->remove_A_set(1, epsilon);
            }

            //debug(z, current_density);
        }
        return make_pair(number_of_passes, best_density);
    };

    void run(){
        long double min_z_candidate = 1.L/sqrtl(max(G->n1, G->n2));
        long double max_z_candidate = sqrtl(max(G->n1, G->n2));

        std::string result_folder_path = "../../Results/";


        string epsilon_string =  to_string(epsilon);
        if(epsilon_string.find('.') != std::string::npos){
            while(epsilon_string[epsilon_string.size()-1] == '0'){
                epsilon_string = epsilon_string.substr(0, epsilon_string.size()-1);
                if(epsilon_string[epsilon_string.size()-1] == '.'){
                    epsilon_string = epsilon_string.substr(0, epsilon_string.size()-1);
                    break;
                }
            }
        }

        std::string delta_string =  std::to_string(delta);
        if(delta_string.find('.') != std::string::npos){
            while(delta_string[delta_string.size()-1] == '0'){
                delta_string = delta_string.substr(0, delta_string.size()-1);
                if(delta_string[delta_string.size()-1] == '.'){
                    delta_string = delta_string.substr(0, delta_string.size()-1);
                    break;
                }
            }
        }

        string result_file_name = algorithm_name + "___" + test_case + "___" + delta_string + "___" + epsilon_string + ".json";

        std::ifstream f(result_folder_path + result_file_name);
        if (f.good()) {
            std::cout << "File already exists. Do you want to clear it? (y/n): ";
            char choice;
            std::cin >> choice;
            if (choice == 'y') {
                std::ofstream ofs(result_folder_path + result_file_name, std::ofstream::trunc);
                ofs.close();
            } else {
                std::cout << "Exiting without clearing the file." << std::endl;
                return;
            }
        }
        f.close();

        std::ofstream resultFile(result_folder_path + result_file_name);
        resultFile << "[";


        for(long double z = min_z_candidate; z <= max_z_candidate; z *= (1.L+epsilon)) {
            std::pair<int, long double> result = run_for_fixed_z(z);   
                if (resultFile.is_open()) {
                    resultFile << "{\n";
                    resultFile << "\"candidate_z\": " << z << ",\n";
                    resultFile << "\"number_of_rounds\": " << result.first << ",\n";
                    resultFile << "\"density\": " << result.second << "\n";
                    resultFile << "},\n";
                } else {
                    std::cerr << "Unable to open file" + result_folder_path + result_file_name + "\n";
                    exit(1);
                }
        }

        resultFile.seekp(-2, std::ios_base::end);
        resultFile << "\n]";
        resultFile.flush();
        resultFile.close();
    }
};


std::vector<std::pair<int, int>*> get_edges(std::string file_path){
    std::vector<std::pair<int, int>*> edges = std::vector<std::pair<int, int>*>();
    std::ifstream file(file_path);

    if (!file.is_open()) {
        std::cerr << "Error: Could not open file " << file_path << std::endl;
        return edges;
    }

    std::string line;
    while (std::getline(file, line)) {
        std::istringstream iss(line);
        int a, b;
        if (iss >> a >> b) {
            edges.push_back(new std::pair<int, int>(a, b));
        } else {
            std::cerr << "Warning: Skipping malformed line: " << line << std::endl;
        }
    }
    file.close();
    return edges;
}

std::pair<int, int>* n_m(std::vector<std::pair<int, int>*> edges){
    std::pair<int, int>* n_m = new std::pair<int, int>(0, edges.size());
    for(int i = 0; i < edges.size(); i++){
        n_m->first = std::max(n_m->first, std::max(edges[i]->first, edges[i]->second));
    }
    return n_m;
}

int main() {
    auto edges = get_edges("../../Datasets/Slashdot0902.txt");  
    pair<int, int>* NandM = n_m(edges);
    GraphBahmani* G = new GraphBahmani(NandM->first + 1, NandM->first + 1);
    AlgorithmBahmani* alg = new AlgorithmBahmani(G, 0.8, 0.6, "Slashdot0902", edges);
    alg->run();
}